<?php

namespace App\Services;

use App\Models\RationCard;
use Illuminate\Support\Facades\Storage;
use Spatie\Browsershot\Browsershot;
use Exception;

class RationCardImageService
{
    protected $fontPath;

    public function __construct()
    {
        // Set font path for HTML rendering
        $this->fontPath = storage_path('fonts/NotoSansTamil.ttf');
    }

    /**
     * Generate ration card image for a given ration card ID
     * Uses HTML to Image conversion for perfect Tamil text rendering
     *
     * @param int $rationCardId
     * @return array Returns ['success' => bool, 'imagePath' => string|null, 'filename' => string, 'error' => string|null]
     */
    public function generateRationCardImage($rationCardId)
    {
        try {
            // Fetch ration card with voters
            $rationCard = RationCard::with(['voters' => function ($query) {
                $query->where('is_deleted', false);
            }])->find($rationCardId);

            if (!$rationCard) {
                return [
                    'success' => false,
                    'imagePath' => null,
                    'filename' => null,
                    'error' => 'Ration card not found'
                ];
            }

            // Get head of family
            $head = $rationCard->voters->where('is_head', 1)->first();
            
            if (!$head) {
                return [
                    'success' => false,
                    'imagePath' => null,
                    'filename' => null,
                    'error' => 'Head of family not found'
                ];
            }

            // Calculate voter counts
            $maleCount = $rationCard->voters->where('gender', 'male')->count();
            $femaleCount = $rationCard->voters->where('gender', 'female')->count();
            $totalCount = $rationCard->voters->count();

            // Generate member unique ID: booth-street-cardNumber
            $boothNumber = $head->booth_number ?? 'N/A';
            $memberUniqueId = "{$rationCard->id}-{$boothNumber}-{$rationCard->ration_card_number}";

            // Get background image path
            $backgroundPath = public_path('assets/imgs/ration_card_bg.jpg');
            
            if (!file_exists($backgroundPath)) {
                return [
                    'success' => false,
                    'imagePath' => null,
                    'filename' => null,
                    'error' => 'Background image not found at: ' . $backgroundPath
                ];
            }

            // Get photo path
            $photoPath = $this->resolvePhotoPath($rationCard->head_photo);

            // Prepare data for the Blade template
            $data = [
                'fontPath' => $this->fontPath,
                'backgroundImage' => $backgroundPath,
                'headName' => $head->name,
                'address' => $rationCard->address,
                'mobile' => $head->mobile_number,
                'maleCount' => $maleCount,
                'femaleCount' => $femaleCount,
                'totalCount' => $totalCount,
                'memberUniqueId' => $memberUniqueId,
                'rationCardNumber' => $rationCard->ration_card_number,
                'photoPath' => $photoPath,
            ];

            // Render HTML from Blade template
            $html = view('ration_card_template', $data)->render();

            // Generate filename
            $filename = "ration_card_{$rationCard->ration_card_number}.png";
            $outputPath = storage_path('app/temp/' . $filename);

            // Ensure temp directory exists
            if (!file_exists(dirname($outputPath))) {
                mkdir(dirname($outputPath), 0755, true);
            }

            // Convert HTML to PNG using Browsershot
            Browsershot::html($html)
                ->setNodeBinary('/Users/mohanakrishnanm/.nvm/versions/node/v20.17.0/bin/node')
                ->setNpmBinary('/Users/mohanakrishnanm/.nvm/versions/node/v20.17.0/bin/npm')
                ->windowSize(1181, 886)
                ->setScreenshotType('png')
                ->save($outputPath);

            return [
                'success' => true,
                'imagePath' => $outputPath,
                'filename' => $filename,
                'error' => null
            ];

        } catch (Exception $e) {
            return [
                'success' => false,
                'imagePath' => null,
                'filename' => null,
                'error' => 'Image generation failed: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Resolve photo path from multiple possible locations
     */
    protected function resolvePhotoPath($photoPath)
    {
        if (!$photoPath) {
            return null;
        }

        $possiblePaths = [
            storage_path('app/public/' . $photoPath),
            storage_path('app/' . $photoPath),
            public_path('storage/' . $photoPath),
        ];

        foreach ($possiblePaths as $path) {
            if (file_exists($path)) {
                return $path;
            }
        }

        return null;
    }

    /**
     * Read image file and return its contents
     *
     * @param string $imagePath Path to the image file
     * @return string|false Image contents or false on failure
     */
    public function readImageFile($imagePath)
    {
        if (file_exists($imagePath)) {
            $contents = file_get_contents($imagePath);
            // Clean up temp file after reading
            @unlink($imagePath);
            return $contents;
        }
        return false;
    }
}
