<?php

namespace App\Repositories;

use App\Models\Street;

class StreetRepository
{
    protected $street;

    public function __construct(Street $street)
    {
        $this->street = $street;
    }

    // Create Street
    public function createStreet(array $data)
    {
        // Check if street with same name already exists (case-insensitive)
        $existingStreet = $this->street::whereRaw('LOWER(street_name) = ?', [strtolower($data['street_name'])])
                                      ->where('is_deleted', 0)
                                      ->first();
        
        if ($existingStreet) {
            return null; // Street name already exists
        }

        $data['is_deleted'] = false; // Ensure new streets are not marked as deleted
        return $this->street::create($data);
    }

    // Get all Streets
    public function getAll()
    {
        return $this->street::where('is_deleted', 0)->get();
    }

    // Get count of all Streets
    public function getStreetCount()
    {
        return $this->street::where('is_deleted', 0)->count();
    }

    // Get Street by ID
    public function getStreetById($streetId)
    {
        return $this->street::where('id', $streetId)->where('is_deleted', 0)->first();
    }

    // Update Street
    public function update($streetId, array $data)
    {
        // Find the street by its unique ID
        $street = $this->street::find($streetId);

        if (!$street) {
            // If the street with the given ID doesn't exist
            return null; // Street not found, so return null
        }

        // If the street_name is being updated, check for conflicts with existing street_names
        if (isset($data['street_name']) && strtolower($data['street_name']) !== strtolower($street->street_name)) {
            // Check if the new street_name already exists in the database (case-insensitive)
            $existingStreet = $this->street::whereRaw('LOWER(street_name) = ?', [strtolower($data['street_name'])])
                                          ->where('is_deleted', 0)
                                          ->where('id', '!=', $streetId)
                                          ->first();
            
            if ($existingStreet) {
                // If the street name already exists, return a conflict
                return null;
            }
        }

        // Update the street with the new data
        $street->update($data);

        return $street; // Return the updated street
    }

    // Soft delete Street
    public function delete($streetId)
    {
        $street = $this->street::where('id', $streetId)->first();

        if ($street && !$street->is_deleted) {
            // Soft delete all voters associated with this street
            \App\Models\Voter::where('street_id', $streetId)
                             ->where('is_deleted', false)
                             ->update(['is_deleted' => true]);
            
            // Soft delete the street
            $street->is_deleted = true;
            return $street->save(); // returns true if updated
        }

        return false; // Street not found or already deleted
    }
}
