<?php

namespace App\Repositories;

use App\Models\Grievance;
use Illuminate\Support\Facades\DB;

class GrievanceRepository
{
    /**
     * Create a new grievance
     */
    public function create(array $data)
    {
        return Grievance::create($data);
    }

    /**
     * Update a grievance by ID
     */
    public function update($id, array $data)
    {
        $grievance = Grievance::find($id);
        if ($grievance) {
            $grievance->update($data);
            return $grievance->fresh();
        }
        return null;
    }

    /**
     * Get grievance by ID
     */
    public function getById($id)
    {
        return Grievance::with(['voter', 'rationCard'])->find($id);
    }

    /**
     * Get all grievances by status, ordered by nearest target_resolve_by date
     */
    public function getAllByStatus($status)
    {
        return Grievance::with(['voter', 'rationCard'])
            ->where('status', $status)
            ->orderByRaw('CASE WHEN target_resolve_by IS NULL THEN 1 ELSE 0 END')
            ->orderBy('target_resolve_by', 'asc')
            ->orderBy('reported_at', 'desc')
            ->get();
    }

    /**
     * Get all grievances (with optional status filter)
     */
    public function getAll($status = null)
    {
        $query = Grievance::with(['voter', 'rationCard']);
        
        if ($status) {
            $query->where('status', $status);
        }

        return $query->orderByRaw('CASE WHEN target_resolve_by IS NULL THEN 1 ELSE 0 END')
            ->orderBy('target_resolve_by', 'asc')
            ->orderBy('reported_at', 'desc')
            ->get();
    }

    /**
     * Get all grievances by voter ID
     */
    public function getByVoterId($voterId)
    {
        return Grievance::with(['voter', 'rationCard'])
            ->where('voter_id', $voterId)
            ->orderBy('reported_at', 'desc')
            ->get();
    }

    /**
     * Get all grievances by ration card number
     */
    public function getByRationCardNumber($rationCardNumber)
    {
        return Grievance::with(['voter', 'rationCard'])
            ->where('ration_card_number', $rationCardNumber)
            ->orderBy('reported_at', 'desc')
            ->get();
    }

    /**
     * Delete a grievance (soft delete if needed)
     */
    public function delete($id)
    {
        $grievance = Grievance::find($id);
        if ($grievance) {
            return $grievance->delete();
        }
        return false;
    }

    /**
     * Get grievance counts by status
     */
    public function getCountsByStatus()
    {
        return Grievance::select('status', DB::raw('count(*) as count'))
            ->groupBy('status')
            ->pluck('count', 'status')
            ->toArray();
    }

    /**
     * Get filtered grievance counts by status based on ration card number
     */
    public function getFilteredCountsByStatus($rationCardNumber = null)
    {
        $query = Grievance::query();

        // Apply the same filtering logic as in getPaginatedGrievances
        if ($rationCardNumber) {
            $query->where(function ($q) use ($rationCardNumber) {
                // Direct match on grievance ration_card_number
                $q->where('ration_card_number', $rationCardNumber)
                  // Or match through voter's ration card relationship
                  ->orWhereHas('voter', function ($voterQuery) use ($rationCardNumber) {
                      $voterQuery->whereHas('rationCard', function ($rcQuery) use ($rationCardNumber) {
                          $rcQuery->where('ration_card_number', $rationCardNumber);
                      });
                  })
                  // Or match through direct rationCard relationship
                  ->orWhereHas('rationCard', function ($rationCardQuery) use ($rationCardNumber) {
                      $rationCardQuery->where('ration_card_number', $rationCardNumber);
                  });
            });
        }

        return $query->select('status', DB::raw('count(*) as count'))
            ->groupBy('status')
            ->pluck('count', 'status')
            ->toArray();
    }

    /**
     * Get grievance counts by status with optional status filter (for getAllPaginated)
     */
    public function getCountsByStatusWithFilter($status = null)
    {
        $query = Grievance::query();
        
        // If a specific status is being filtered, we still want to show all status counts
        // but this method allows for future flexibility if needed
        
        return $query->select('status', DB::raw('count(*) as count'))
            ->groupBy('status')
            ->pluck('count', 'status')
            ->toArray();
    }

    /**
     * Get paginated grievances with optional ration card search
     */
    public function getPaginatedGrievances($rationCardNumber = null, $perPage = 15, $status = null)
    {
        $query = Grievance::with(['voter.rationCard', 'rationCard']);

        // If ration card number is provided, search in grievances and related voters
        if ($rationCardNumber) {
            $query->where(function ($q) use ($rationCardNumber) {
                // Direct match on grievance ration_card_number
                $q->where('ration_card_number', $rationCardNumber)
                  // Or match through voter's ration card relationship
                  ->orWhereHas('voter', function ($voterQuery) use ($rationCardNumber) {
                      $voterQuery->whereHas('rationCard', function ($rcQuery) use ($rationCardNumber) {
                          $rcQuery->where('ration_card_number', $rationCardNumber);
                      });
                  })
                  // Or match through direct rationCard relationship
                  ->orWhereHas('rationCard', function ($rationCardQuery) use ($rationCardNumber) {
                      $rationCardQuery->where('ration_card_number', $rationCardNumber);
                  });
            });
        }

        // Filter by status if provided
        if ($status) {
            $query->where('status', $status);
        }

        // Order by target resolve date and reported date
        $query->orderByRaw('CASE WHEN target_resolve_by IS NULL THEN 1 ELSE 0 END')
              ->orderBy('target_resolve_by', 'asc')
              ->orderBy('reported_at', 'desc');

        return $query->paginate($perPage);
    }

    /**
     * Get all grievances paginated (with optional status filter)
     */
    public function getAllPaginated($status = null, $perPage = 15)
    {
        $query = Grievance::with(['voter', 'rationCard']);
        
        if ($status) {
            $query->where('status', $status);
        }

        return $query->orderByRaw('CASE WHEN target_resolve_by IS NULL THEN 1 ELSE 0 END')
            ->orderBy('target_resolve_by', 'asc')
            ->orderBy('reported_at', 'desc')
            ->paginate($perPage);
    }
}
