<?php

namespace App\Services;

use App\Jobs\ProcessVoterImagePage;
use App\Models\ImagePageImportLog;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File;

class VoterImageImportService
{
    /**
     * Scan a directory for page images and dispatch jobs.
     * @param string $directory Absolute or relative path to Output folder.
     * @param string|null $boothNumber Optional booth number context.
     * @param int|null $limit Optional limit for testing.
     */
    public function dispatchForDirectory(string $directory, ?string $boothNumber = null, ?int $limit = null): array
    {
        $path = rtrim($directory, '/');
        if (!is_dir($path)) {
            return ['success' => false, 'error' => 'Directory not found: '.$path];
        }

        $extensions = ['jpg','jpeg','png'];
        $files = collect(File::files($path))
            ->filter(fn($f) => in_array(strtolower($f->getExtension()), $extensions))
            ->sortBy(fn($f) => $f->getFilename())
            ->values();

        if ($limit) {
            $files = $files->take($limit);
        }

        $dispatched = 0; $logs = [];
        foreach ($files as $file) {
            $log = ImagePageImportLog::create(['page_path' => $file->getRealPath(), 'status' => 'pending']);
            ProcessVoterImagePage::dispatch($log, $file->getRealPath(), $boothNumber);
            $logs[] = $log->id; $dispatched++;
        }

        Log::info('Dispatched image page import jobs', ['count' => $dispatched, 'directory' => $path]);

        return [
            'success' => true,
            'directory' => $path,
            'jobs_dispatched' => $dispatched,
            'log_ids' => $logs,
        ];
    }
}
