<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\WhatsAppService;
use Illuminate\Support\Facades\Log;

class WhatsAppController extends Controller
{
    protected WhatsAppService $whatsAppService;

    public function __construct(WhatsAppService $whatsAppService)
    {
        $this->whatsAppService = $whatsAppService;
    }

    public function sendPlanExpiry(Request $request)
    {
        try {
            $data = $request->validate([
                'mobile_number'   => 'required|string',
                'name'            => 'required|string',
                'plan_name'       => 'required|string',
                'expire_in_days'  => 'required|string',
            ]);

            // Check if WhatsApp service is configured
            if (!$this->whatsAppService->isConfigured()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'WhatsApp service is not properly configured. Please check your environment variables.',
                ], 500);
            }

            $response = $this->whatsAppService->sendPlanExpireReminder($data);

            return response()->json([
                'status'   => 'success',
                'whatsapp' => $response,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to send WhatsApp message: ' . $e->getMessage(),
            ], 500);
        }
    }

public function sendSimpleMessage(Request $request, WhatsAppService $whatsapp)
{
    try {
        // Get the JSON content and decode it
        $content = $request->getContent();
        $data = json_decode($content, true);
        
        // Log incoming request for debugging
        Log::info('WhatsApp request DEBUG:', [
            'method' => $request->method(),
            'url' => $request->fullUrl(),
            'raw_content' => $content,
            'content_type' => $request->header('Content-Type'),
            'decoded_data' => $data,
            'request_all' => $request->all(),
            'is_json' => $request->isJson(),
            'json_error' => json_last_error_msg()
        ]);
        
        // If JSON decode failed or no data, try request->all()
        if (empty($data)) {
            $data = $request->all();
        }
        
        // Manual validation to provide better error messages
        if (empty($data['mobile_number'])) {
            return response()->json([
                'status' => 'error',
                'message' => 'mobile_number field is required',
                'received_data' => $data
            ], 422);
        }
        
        if (empty($data['message'])) {
            return response()->json([
                'status' => 'error', 
                'message' => 'message field is required',
                'received_data' => $data
            ], 422);
        }
        
        $validated = [
            'mobile_number' => $data['mobile_number'],
            'message' => $data['message']
        ];

        // Check if WhatsApp service is configured
        if (!$whatsapp->isConfigured()) {
            return response()->json([
                'status' => 'error',
                'message' => 'WhatsApp service is not properly configured. Please check your environment variables.',
            ], 500);
        }

        $response = $whatsapp->sendTextMessage(
            $validated['mobile_number'],
            $validated['message']
        );

        return response()->json([
            'status' => 'success',
            'whatsapp' => $response,
        ]);
    } catch (\Illuminate\Validation\ValidationException $e) {
        return response()->json([
            'status' => 'error',
            'message' => 'Validation failed: ' . implode(', ', $e->validator->errors()->all()),
            'errors' => $e->validator->errors()
        ], 422);
    } catch (\Exception $e) {
        return response()->json([
            'status' => 'error',
            'message' => 'Failed to send WhatsApp message: ' . $e->getMessage(),
        ], 500);
    }
}

public function sendTemplateTest(Request $request, WhatsAppService $whatsapp)
{
    $mobile = $request->input('mobile_number');

    $response = $whatsapp->sendHelloTemplate($mobile);

    return response()->json([
        'status'   => 'success',
        'whatsapp' => $response,
    ]);
}



/**
 * Test WhatsApp configuration
 */
public function testConfiguration()
{
    return response()->json([
        'whatsapp_configured' => $this->whatsAppService->isConfigured(),
        'phone_number_id' => config('whatsapp.phone_number_id') ? 'Set' : 'Not set',
        'access_token' => config('whatsapp.access_token') ? 'Set (' . strlen(config('whatsapp.access_token')) . ' chars)' : 'Not set',
        'api_version' => config('whatsapp.api_version'),
    ]);
}

/**
 * Debug endpoint - echo back exactly what we receive
 */
public function debugRequest(Request $request)
{
    $content = $request->getContent();
    $data = json_decode($content, true);
    
    return response()->json([
        'method' => $request->method(),
        'url' => $request->fullUrl(),
        'headers' => $request->headers->all(),
        'raw_content' => $content,
        'decoded_json' => $data,
        'request_all' => $request->all(),
        'is_json' => $request->isJson(),
        'content_type' => $request->header('Content-Type'),
        'json_error' => json_last_error_msg(),
        'has_mobile' => isset($data['mobile_number']) ? 'YES' : 'NO',
        'has_message' => isset($data['message']) ? 'YES' : 'NO',
    ]);
}

public function sendBulkTemplate(Request $request, WhatsAppService $whatsapp)
{
    try {
        // Get the JSON content and decode it
        $content = $request->getContent();
        $data = json_decode($content, true);
        
        // If JSON decode failed or no data, try request->all()
        if (empty($data)) {
            $data = $request->all();
        }

        // Log the received data for debugging
        \Illuminate\Support\Facades\Log::info('Bulk Template Request Data:', $data);

        // Manual validation for better error messages
        if (empty($data['template_name'])) {
            return response()->json([
                'status' => 'error',
                'message' => 'template_name field is required',
                'received_data' => $data,
                'expected_format' => [
                    'template_name' => 'your_template_name',
                    'language_code' => 'en_US (optional)',
                    'recipients' => [
                        [
                            'mobile_number' => '919xxxxxxxxx',
                            'body_params' => ['param1', 'param2']
                        ]
                    ]
                ]
            ], 422);
        }

        if (empty($data['recipients']) || !is_array($data['recipients'])) {
            return response()->json([
                'status' => 'error',
                'message' => 'recipients field is required and must be an array',
                'received_data' => $data
            ], 422);
        }

        $templateName = $data['template_name'];
        $languageCode = $data['language_code'] ?? 'en_US';
        $recipients   = $data['recipients'];

        // Validate each recipient
        foreach ($recipients as $index => $recipient) {
            if (empty($recipient['mobile_number'])) {
                return response()->json([
                    'status' => 'error',
                    'message' => "recipients[{$index}].mobile_number is required",
                    'received_data' => $data
                ], 422);
            }
        }

        if (!$whatsapp->isConfigured()) {
            return response()->json([
                'status'  => 'error',
                'message' => 'WhatsApp service not configured. Check .env.',
            ], 500);
        }

        $results = $whatsapp->sendBulkTemplateMessages(
            $recipients,
            $templateName,
            $languageCode
        );

        return response()->json([
            'status'  => 'success',
            'template_name' => $templateName,
            'language_code' => $languageCode,
            'total_recipients' => count($recipients),
            'results' => $results,
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'status' => 'error',
            'message' => 'Failed to send bulk template messages: ' . $e->getMessage(),
        ], 500);
    }
}


}
