<?php

namespace App\Http\Controllers;

use App\Services\GrievanceService;
use Illuminate\Http\Request;

class GrievanceController extends Controller
{
    protected $grievanceService;

    public function __construct(GrievanceService $grievanceService)
    {
        $this->grievanceService = $grievanceService;
    }

    /**
     * Create a new grievance
     * POST /api/grievance/create
     */
    public function create(Request $request)
    {
        return $this->grievanceService->createGrievance($request->all());
    }

    /**
     * Update an existing grievance
     * PUT /api/grievance/update/{id}
     */
    public function update(Request $request, $id)
    {
        return $this->grievanceService->updateGrievance($id, $request->all());
    }

    /**
     * Get a single grievance by ID
     * GET /api/grievance/get/{id}
     */
    public function getById($id)
    {
        return $this->grievanceService->getGrievanceById($id);
    }

    /**
     * Get all grievances with optional status filter
     * GET /api/grievance/getAll?status=OPEN&per_page=20
     */
    public function getAll(Request $request)
    {
        $status = $request->query('status');
        $perPage = $request->input('per_page', 20);
        return $this->grievanceService->getAllGrievances($status, $perPage);
    }

    /**
     * Get all grievances by status (ordered by nearest target_resolve_by date)
     * GET /api/grievance/getByStatus/{status}?per_page=20
     */
    public function getByStatus(Request $request, $status)
    {
        $perPage = $request->input('per_page', 20);
        return $this->grievanceService->getAllByStatus($status, $perPage);
    }

    /**
     * Get all grievances by voter ID
     * GET /api/grievance/getByVoterId/{voterId}?per_page=20
     */
    public function getByVoterId(Request $request, $voterId)
    {
        $perPage = $request->input('per_page', 20);
        return $this->grievanceService->getByVoterId($voterId, $perPage);
    }

    /**
     * Get all grievances by ration card number
     * GET /api/grievance/getByRationCard/{rationCardNumber}?per_page=20
     */
    public function getByRationCardNumber(Request $request, $rationCardNumber)
    {
        $perPage = $request->input('per_page', 20);
        return $this->grievanceService->getByRationCardNumber($rationCardNumber, $perPage);
    }

    /**
     * Delete a grievance
     * DELETE /api/grievance/delete/{id}
     */
    public function delete($id)
    {
        return $this->grievanceService->deleteGrievance($id);
    }

    /**
     * Get available status values
     * GET /api/grievance/statuses
     */
    public function getStatuses()
    {
        return response()->json([
            'status' => 200,
            'message' => 'Available statuses',
            'data' => [
                'statuses' => \App\Models\Grievance::getStatuses(),
            ],
        ], 200);
    }

    /**
     * Get paginated grievances with optional ration card search
     * GET /api/grievance/paginated?ration_card_number=123456&per_page=15&status=OPEN&page=1
     */
    public function getPaginatedGrievances(Request $request)
    {
        $rationCardNumber = $request->query('ration_card_number');
        $perPage = (int) $request->query('per_page', 15);
        $status = $request->query('status');

        return $this->grievanceService->getPaginatedGrievances($rationCardNumber, $perPage, $status);
    }

    /**
     * Get all grievances paginated
     * GET /api/grievance/paginated-all?status=OPEN&per_page=15&page=1
     */
    public function getAllPaginated(Request $request)
    {
        $status = $request->query('status');
        $perPage = (int) $request->query('per_page', 15);

        return $this->grievanceService->getAllGrievancesPaginated($status, $perPage);
    }
}
