<?php

namespace App\Http\Controllers;

use App\Models\Voter;
use App\Models\Booth;
use App\Models\Street;
use App\Services\BoothService;
use App\Services\VoterService; // Assuming you have a VoterService
use App\Services\RationCardService; // Assuming you have a RationCardService
use App\ResponseHandler\Response;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    protected $boothService;
    protected $voterService; // Add voter service
    protected $rationCardService; // Add ration card service
    protected $response;

    public function __construct(
        BoothService $boothService,
        RationCardService $rationCardService, // Inject ration card service
        VoterService $voterService, // Inject voter service
        Response $response
    ) {
        $this->boothService = $boothService;
        $this->voterService = $voterService;
        $this->rationCardService = $rationCardService;
        $this->response = $response;
    }

    // Add this method to fetch all counts
    public function getDashboardCounts()
    {
        try {
            $voterCount = $this->voterService->getVoterCount();
            $rationCardCount = $this->rationCardService->getRationCardCount();
            $boothCount = $this->boothService->getBoothCount();
            $unmappedVotersCount = Voter::where('is_deleted', false)
                ->whereNull('ration_card_id')
                ->count();

            return response()->json([
                'status' => 200,
                'message' => 'Data fetched successfully',
                'data' => [
                    'voter_count' => $voterCount,
                    'ration_card_count' => $rationCardCount,
                    'booth_count' => $boothCount,
                    'unmapped_voters_count' => $unmappedVotersCount
                ]
            ], 200);
        } catch (\Exception $e) {
            return $this->response->respondWithError('Error fetching dashboard counts: ' . $e->getMessage());
        }
    }

    /**
     * Get all booths with voters (booth-wise and street-wise) with pagination
     */
    public function getAllBoothsWithVoters()
    {
        try {
            $perPage = request()->input('per_page', 15);
            
            // Get paginated booths
            $boothsPaginated = Booth::where('is_deleted', false)
                ->paginate($perPage);

            // Transform booth data
            $booths = $boothsPaginated->getCollection()->map(function ($booth) {
                // Get total voters count in this booth
                $totalVotersInBooth = Voter::where('booth_number', $booth->booth_number)
                    ->where('is_deleted', false)
                    ->count();

                // Get only streets that have voters in this specific booth
                $streetsInBooth = Street::whereExists(function($query) use ($booth) {
                    $query->select(DB::raw(1))
                          ->from('voters')
                          ->whereRaw('voters.street_id = streets.id')
                          ->where('voters.booth_number', $booth->booth_number)
                          ->where('voters.is_deleted', false);
                })->where('is_deleted', false)->get();

                // Get streets in this booth with voter counts
                $streets = $streetsInBooth->map(function ($street) use ($booth) {
                    // Get total voters count in this street of this booth
                    $totalVotersInStreet = Voter::where('booth_number', $booth->booth_number)
                        ->where('street_id', $street->id)
                        ->where('is_deleted', false)
                        ->count();

                    // Only include streets that have voters
                    if ($totalVotersInStreet > 0) {
                        return [
                            'id' => $street->id,
                            'street_name' => $street->street_name,
                            'total_voters' => $totalVotersInStreet,
                        ];
                    }
                    return null;
                })->filter()->values(); // Remove null entries and reset keys

                return [
                    'id' => $booth->id,
                    'booth_number' => $booth->booth_number,
                    'booth_address' => $booth->booth_address,
                    'total_voters' => $totalVotersInBooth,
                    'streets' => $streets,
                    'street_count' => $streets->count(),
                ];
            });

            // Replace the collection in paginator
            $boothsPaginated->setCollection($booths);

            return response()->json([
                'status' => 200,
                'message' => 'Booths with voter data fetched successfully',
                'data' => [
                    'booths' => $boothsPaginated->items(),
                    'pagination' => [
                        'current_page' => $boothsPaginated->currentPage(),
                        'last_page' => $boothsPaginated->lastPage(),
                        'per_page' => $boothsPaginated->perPage(),
                        'total' => $boothsPaginated->total(),
                        'from' => $boothsPaginated->firstItem(),
                        'to' => $boothsPaginated->lastItem(),
                    ],
                ],
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'Failed to fetch booth data',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Download total unmapped voters as Excel (all booths)
     */
    public function downloadTotalUnmappedVotersAsExcel()
    {
        try {
            return $this->rationCardService->downloadTotalUnmappedVotersAsExcel();
        } catch (\Exception $e) {
            return $this->response->respondWithError('Failed to generate total unmapped voters Excel export: ' . $e->getMessage(), 500);
        }
    }
}
