<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\BoothService;
use App\ResponseHandler\Response;
use App\Security\DecryptUtils;
use App\Validations\Validation;

class BoothController extends Controller
{
    protected $boothService;
    protected $response;

    public function __construct(BoothService $boothService, Response $response)
    {
        $this->boothService = $boothService;
        $this->response = $response;
    }

    // Create a Booth
    // public function createBooth(Request $request)
    // {
    //     $data = $request->all();

    //     if (empty($data)) {
    //         return $this->response->respondWithError('Request data is empty');
    //     }

    //     return $this->boothService->createBooth($data);
    // }

    public function createBooth(Request $request)
    {
        // Step 1: Get raw content (encrypted JSON)
        $data = $request->getContent();

        if (empty($data)) {
            return $this->response->respondWithError('Request data is empty');
        }

        // Step 2: Decrypt the incoming data
        $decryptedData = DecryptUtils::decryptRequestData($data);

        if ($decryptedData == null || !is_array($decryptedData)) {
            return $this->response->respondWithError('Invalid or empty request data');
        }

        // Step 3: Validate decrypted data
        $validationResponse = Validation::validationForBooth($decryptedData);
        if ($validationResponse->getStatusCode() !== 200) {
            return $validationResponse;
        }

        // Step 4: Pass decrypted data to BoothService
        return $this->boothService->createBooth($decryptedData);
    }


    // Edit Booth
    // public function editBooth(Request $request, $booth_number)
    // {
    //     $data = $request->all();

    //     if (empty($data)) {
    //         return $this->response->respondWithError('Request data is empty');
    //     }

    //     return $this->boothService->updateBooth($booth_number, $data);
    // }

    public function editBooth(Request $request, $boothId)
    {
        // $data = $request->all();

        // if (empty($data)) {
        //     return $this->response->respondWithError('Request data is empty');
        // }


        // Step 1: Get raw encrypted JSON
        $data = $request->getContent();

        if (empty($data)) {
            return $this->response->respondWithError('Request data is empty');
        }

        // Step 2: Decrypt
        $decryptedData = DecryptUtils::decryptRequestData($data);

        if ($decryptedData == null || !is_array($decryptedData)) {
            return $this->response->respondWithError('Invalid or empty request data');
        }

        // Step 3: Validate just like create
        $validationResponse = Validation::validationForBooth($decryptedData);
        if ($validationResponse->getStatusCode() !== 200) {
            return $validationResponse;
        }

        // Call the service to update the booth by its unique ID and pass the new data
        return $this->boothService->updateBooth($boothId, $decryptedData);
    }


    // Get All Booths and Count
    public function getAllBooths()
    {
        return $this->boothService->getAllBooths();
    }

    // Get booths along with streets, categories and castes in one response
    public function getBoothsWithLists()
    {
        return $this->boothService->getBoothsWithLists();
    }

    // Get static lists (streets, categories, castes, genders)
    public function getStaticList()
    {
        return $this->boothService->getStaticList();
    }

    // Get Single Booth by ID
    public function getBooth($boothId)
    {
        if (empty($boothId)) {
            return $this->response->respondWithError('Booth ID is required');
        }

        return $this->boothService->getBoothById($boothId);
    }

    // Delete Booth (soft delete)
    public function deleteBooth($booth_number)
    {
        if (empty($booth_number)) {
            return $this->response->respondWithError('Booth number is required');
        }

        // Directly return the response from the service
        return $this->boothService->deleteBooth($booth_number);
    }

    public function downloadAsExcel()
    {

        // Directly return the response from the service
        return $this->boothService->downloadAsExcel();
    }
}
