# Voter Management API Documentation

## Create Voter Endpoint

**URL:** `POST /api/voters/create`

**Description:** Creates a standalone voter with optional ration card association

**Authentication:** Required (if applicable)

**Content-Type:** `application/json`

---

## Request Format

### Headers
```
Content-Type: application/json
Accept: application/json
```

### Request Body (Encrypted JSON)

The request body should be encrypted using your existing encryption method. Below are the decrypted payload examples:

#### Minimal Required Data
```json
{
  "voter_id_number": "VTR1234567890",
  "name": "John Doe",
  "gender": "Male"
}
```

#### Complete Data Example
```json
{
  "ration_card_number": "RCN1234567890",
  "voter_id_number": "VTR1234567890", 
  "name": "John Doe",
  "gender": "Male",
  "year_of_birth": 1990,
  "date_of_birth": "1990-05-15",
  "anniversary_date": "2015-12-10",
  "mobile_number": "9876543210",
  "aadhar_number": "123456789012",
  "booth_id": 1,
  "booth_number": "001A",
  "street_id": 5,
  "street_name": "Main Street",
  "is_head": true,
  "voter_image": null
}
```

#### With Optional Fields
```json
{
  "voter_id_number": "VTR9876543210",
  "name": "Jane Smith", 
  "gender": "Female",
  "year_of_birth": 1985,
  "date_of_birth": "1985-08-22",
  "mobile_number": "8765432109",
  "aadhar_number": "987654321098",
  "booth_number": "002B",
  "street_name": "Park Avenue",
  "is_head": false
}
```

---

## Field Descriptions

| Field | Type | Required | Description | Validation Rules |
|-------|------|----------|-------------|------------------|
| `ration_card_number` | string | No | Ration card number (optional) | Must exist in database if provided |
| `voter_id_number` | string | Yes | Unique voter identification | Required, must be unique |
| `name` | string | Yes | Voter's full name | Required, A-Z and spaces only |
| `gender` | string | Yes | Voter's gender | Required, Must be "Male", "Female", or "Other" |
| `year_of_birth` | integer | No | Birth year | Between 1900 and current year |
| `date_of_birth` | string | No | Full birth date | Format: YYYY-MM-DD |
| `anniversary_date` | string | No | Anniversary date | Format: YYYY-MM-DD |
| `mobile_number` | string | No | 10-digit mobile number | Must be exactly 10 digits |
| `aadhar_number` | string | No | 12-digit Aadhar number | Must be exactly 12 digits |
| `booth_id` | integer | No | Booth database ID | Must exist in booths table |
| `booth_number` | string | No | Booth number/code | - |
| `street_id` | integer | No | Street database ID | Must exist in streets table |
| `street_name` | string | No | Street name | - |
| `is_head` | boolean | No | Family head status | Default: false |
| `voter_image` | string | No | Image file path | Set via image upload API |

---

## Success Response Examples

### Successful Creation
```json
{
  "status": 200,
  "message": "Voter created successfully",
  "data": {
    "id": 123,
    "ration_card_id": 45,
    "voter_id_number": "VTR1234567890",
    "name": "John Doe",
    "gender": "Male",
    "year_of_birth": 1990,
    "date_of_birth": "1990-05-15",
    "anniversary_date": "2015-12-10",
    "is_head": true,
    "booth_id": 1,
    "booth_number": "001A",
    "mobile_number": "9876543210",
    "aadhar_number": "123456789012",
    "street_id": 5,
    "street_name": "Main Street",
    "voter_image": null,
    "created_at": "2025-11-18T10:30:00.000000Z",
    "updated_at": "2025-11-18T10:30:00.000000Z"
  }
}
```

### Creation Without Ration Card
```json
{
  "status": 200,
  "message": "Voter created successfully", 
  "data": {
    "id": 124,
    "ration_card_id": null,
    "voter_id_number": "VTR9876543210",
    "name": "Jane Smith",
    "gender": "Female",
    "year_of_birth": 1985,
    "date_of_birth": "1985-08-22",
    "anniversary_date": null,
    "is_head": false,
    "booth_id": null,
    "booth_number": "002B", 
    "mobile_number": "8765432109",
    "aadhar_number": "987654321098",
    "street_id": null,
    "street_name": "Park Avenue",
    "voter_image": null,
    "created_at": "2025-11-18T10:35:00.000000Z",
    "updated_at": "2025-11-18T10:35:00.000000Z"
  }
}
```

---

## Error Response Examples

### Validation Errors

#### Missing Required Field
```json
{
  "status": 422,
  "message": "Please enter Voter ID Number."
}
```

#### Invalid Gender
```json
{
  "status": 422,
  "message": "Gender must be Male, Female, or Other."
}
```

#### Invalid Mobile Number
```json
{
  "status": 422,
  "message": "Mobile Number must be 10 digits."
}
```

#### Invalid Name Format
```json
{
  "status": 422,
  "message": "Name must contain only letters and spaces."
}
```

#### Invalid Date Format
```json
{
  "status": 422,
  "message": "Date of Birth must be in YYYY-MM-DD format."
}
```

### Business Logic Errors

#### Duplicate Voter ID
```json
{
  "status": 409,
  "message": "Voter ID already exists"
}
```

#### Invalid Ration Card
```json
{
  "status": 404,
  "message": "Ration card number does not exist"
}
```

#### Empty Request Data
```json
{
  "status": 400,
  "message": "Request data is empty"
}
```

#### Invalid Request Format
```json
{
  "status": 400,
  "message": "Invalid or empty request data"
}
```

---

## cURL Examples

### Basic Creation
```bash
curl -X POST "http://localhost/api/voters/create" \
  -H "Content-Type: application/json" \
  -d '{
    "voter_id_number": "VTR1234567890",
    "name": "John Doe", 
    "gender": "Male"
  }'
```

### Complete Data Creation
```bash
curl -X POST "http://localhost/api/voters/create" \
  -H "Content-Type: application/json" \
  -d '{
    "ration_card_number": "RCN1234567890",
    "voter_id_number": "VTR1234567890",
    "name": "John Doe",
    "gender": "Male", 
    "date_of_birth": "1990-05-15",
    "mobile_number": "9876543210",
    "aadhar_number": "123456789012",
    "booth_number": "001A",
    "street_name": "Main Street",
    "is_head": true
  }'
```

---

## Related Endpoints

### Get Form Data (Booths, Streets, Genders)
**URL:** `GET /api/voters/getFormData`

**Response:**
```json
{
  "status": 200,
  "message": "Voter form data fetched successfully",
  "data": {
    "booths": [
      {
        "id": 1,
        "booth_number": "001",
        "booth_name": "Booth 001"
      },
      {
        "id": 2,
        "booth_number": "002", 
        "booth_name": "Booth 002"
      }
    ],
    "streets": [
      {
        "id": 1,
        "street_name": "Main Street"
      },
      {
        "id": 2,
        "street_name": "Park Avenue"
      }
    ],
    "genders": ["Male", "Female", "Other"]
  }
}
```

### Get Booths Only
**URL:** `GET /api/voters/getBooths`

**Response:**
```json
{
  "status": 200,
  "message": "Booths fetched successfully",
  "data": [
    {
      "id": 1,
      "booth_number": "001",
      "booth_name": "Booth 001"
    },
    {
      "id": 2,
      "booth_number": "002",
      "booth_name": "Booth 002"
    }
  ]
}
```

### Get Streets Only
**URL:** `GET /api/voters/getStreets`

**Response:**
```json
{
  "status": 200,
  "message": "Streets fetched successfully",
  "data": [
    {
      "id": 1,
      "street_name": "Main Street"
    },
    {
      "id": 2,
      "street_name": "Park Avenue"
    }
  ]
}
```

### Get Voter
**URL:** `GET /api/voters/get/{voterId}`

**Response:**
```json
{
  "status": 200,
  "message": "Voter retrieved successfully",
  "data": {
    "id": 123,
    "voter_id_number": "VTR1234567890",
    "name": "John Doe",
    // ... other fields
  }
}
```

### Update Voter  
**URL:** `PUT /api/voters/update/{voterId}`

**Request:** Same format as create (all fields optional except voter_id_number)

### Delete Voter
**URL:** `DELETE /api/voters/delete/{voterId}`

**Response:**
```json
{
  "status": 200,
  "message": "Voter deleted successfully",
  "data": null
}
```

### Upload Voter Image
**URL:** `POST /api/imageHandler/upload-voter-image`

**Request:** Form-data with `image` file and `voter_id`

**Response:**
```json
{
  "status": 200,
  "message": "Voter image uploaded successfully!",
  "path": "voter_uploads/image_name.jpg",
  "voter": {
    "id": 123,
    "voter_image": "voter_uploads/image_name.jpg",
    // ... other fields
  }
}
```

---

## Notes

1. **Encryption:** All request data must be encrypted using your existing encryption method
2. **Optional Association:** Ration card number is completely optional - voters can be created independently
3. **Image Upload:** Voter images are handled separately via the image upload endpoint
4. **Validation:** Server-side validation ensures data integrity and format compliance
5. **Soft Deletes:** Deleted voters are marked as `is_deleted = true` rather than being permanently removed
6. **Unique Constraints:** Voter ID numbers must be unique across the system

## Status Codes

- **200:** Success
- **400:** Bad Request (invalid data format)
- **404:** Not Found (ration card doesn't exist)
- **409:** Conflict (duplicate voter ID)
- **422:** Validation Error (invalid field values)
- **500:** Server Error