# PDF Import System Architecture

## System Flow Diagram

```
┌─────────────────────────────────────────────────────────────────────┐
│                         USER/CLIENT                                  │
│                     (Web/Mobile/Postman)                            │
└────────────────────────────┬────────────────────────────────────────┘
                             │
                             │ Upload PDF (max 20MB)
                             │
                             ▼
┌─────────────────────────────────────────────────────────────────────┐
│                    API LAYER (Laravel)                               │
│  ┌─────────────────────────────────────────────────────────────┐   │
│  │         VoterPdfImportController.php                        │   │
│  │                                                              │   │
│  │  ├─ uploadPdf()           Upload & queue processing         │   │
│  │  ├─ analyzePdf()          Test PDF structure               │   │
│  │  ├─ getImportStatus()     Check progress                   │   │
│  │  ├─ getAllImports()       List history                     │   │
│  │  ├─ getImportStatistics() Overall stats                    │   │
│  │  ├─ reprocessImport()     Retry failed                     │   │
│  │  ├─ deleteImport()        Remove import                    │   │
│  │  └─ downloadPdf()         Get original file                │   │
│  └─────────────────────────────────────────────────────────────┘   │
└────────────────────────────┬────────────────────────────────────────┘
                             │
                ┌────────────┴────────────┐
                │                         │
                ▼                         ▼
┌───────────────────────┐    ┌────────────────────────┐
│  IMMEDIATE PROCESS    │    │   QUEUE JOB            │
│  (Testing/Small PDFs) │    │   (Production/Large)   │
│                       │    │                        │
│  • Validates file     │    │  • Database queue      │
│  • Stores PDF         │    │  • 3 retry attempts    │
│  • Processes now      │    │  • 1 hour timeout      │
│  • Returns result     │    │  • Background process  │
└───────────┬───────────┘    └────────────┬───────────┘
            │                             │
            └──────────┬──────────────────┘
                       │
                       ▼
┌─────────────────────────────────────────────────────────────────────┐
│                 STORAGE LAYER                                        │
│  ┌──────────────────────────────────────────────────────────────┐  │
│  │  storage/app/pdf-imports/                                    │  │
│  │  └── {uuid}_{timestamp}.pdf                                  │  │
│  │                                                               │  │
│  │  Features:                                                    │  │
│  │  • UUID-based naming (prevents conflicts)                    │  │
│  │  • Outside public directory (secure)                         │  │
│  │  • Tracks original filename                                  │  │
│  └──────────────────────────────────────────────────────────────┘  │
└────────────────────────────┬────────────────────────────────────────┘
                             │
                             ▼
┌─────────────────────────────────────────────────────────────────────┐
│                 SERVICE LAYER                                        │
│  ┌──────────────────────────────────────────────────────────────┐  │
│  │         VoterPdfImportService.php                            │  │
│  │                                                               │  │
│  │  Step 1: Parse PDF                                           │  │
│  │  ├─ Use smalot/pdfparser                                     │  │
│  │  ├─ Extract text from all pages                              │  │
│  │  └─ Split into lines                                         │  │
│  │                                                               │  │
│  │  Step 2: Detect Patterns                                     │  │
│  │  ├─ Pattern 1: Serial + EPIC + Name + Gender + Year         │  │
│  │  ├─ Pattern 2: EPIC + Name + Age + Gender                   │  │
│  │  ├─ Pattern 3: Name - EPIC - Age - Gender                   │  │
│  │  └─ Booth numbers: "Booth No: 123"                          │  │
│  │                                                               │  │
│  │  Step 3: Extract Voters                                      │  │
│  │  ├─ Regex matching per pattern                               │  │
│  │  ├─ Normalize gender (M/F/O)                                │  │
│  │  ├─ Calculate year from age                                  │  │
│  │  └─ Link to booth if exists                                  │  │
│  │                                                               │  │
│  │  Step 4: Batch Processing                                    │  │
│  │  ├─ Process 100 voters per batch                            │  │
│  │  ├─ Transaction per batch                                    │  │
│  │  ├─ Handle duplicates (update existing)                      │  │
│  │  └─ Track errors per voter                                   │  │
│  └──────────────────────────────────────────────────────────────┘  │
└────────────────────────────┬────────────────────────────────────────┘
                             │
                             ▼
┌─────────────────────────────────────────────────────────────────────┐
│                 DATA LAYER (Eloquent Models)                         │
│  ┌──────────────────────────────────────────────────────────────┐  │
│  │  PdfImportLog Model                                          │  │
│  │  ├─ Tracks import progress                                   │  │
│  │  ├─ Stores statistics                                        │  │
│  │  ├─ Records errors                                           │  │
│  │  └─ Status: pending → processing → completed/failed         │  │
│  └──────────────────────────────────────────────────────────────┘  │
│  ┌──────────────────────────────────────────────────────────────┐  │
│  │  Voter Model                                                 │  │
│  │  ├─ voter_id_number (UNIQUE)                                │  │
│  │  ├─ name, gender, year_of_birth                             │  │
│  │  ├─ booth_id, booth_number                                  │  │
│  │  └─ Duplicate check & update                                │  │
│  └──────────────────────────────────────────────────────────────┘  │
└────────────────────────────┬────────────────────────────────────────┘
                             │
                             ▼
┌─────────────────────────────────────────────────────────────────────┐
│                 DATABASE LAYER (MySQL)                               │
│  ┌──────────────────────────────────────────────────────────────┐  │
│  │  pdf_import_logs table                                       │  │
│  │  ├─ id, original_filename, stored_filename                   │  │
│  │  ├─ status, total_voters, imported_voters                    │  │
│  │  ├─ error_message, import_summary (JSON)                     │  │
│  │  └─ timestamps (started_at, completed_at)                    │  │
│  └──────────────────────────────────────────────────────────────┘  │
│  ┌──────────────────────────────────────────────────────────────┐  │
│  │  voters table                                                │  │
│  │  ├─ id, voter_id_number (UNIQUE INDEX)                      │  │
│  │  ├─ name, gender, year_of_birth                             │  │
│  │  ├─ booth_id, booth_number                                  │  │
│  │  └─ ration_card_id, street_id                               │  │
│  └──────────────────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────────────────┘


## Data Flow Example

┌─────────────────────────────────────────────────────────────────────┐
│ Example: Processing voter_list.pdf with 2000 voters                 │
└─────────────────────────────────────────────────────────────────────┘

1. UPLOAD
   ├─ File: voter_list.pdf (5MB)
   ├─ Validation: ✓ PDF, ✓ <20MB
   ├─ Storage: pdf-imports/abc123_1699430400.pdf
   └─ Log: ID=1, status=pending

2. QUEUE
   ├─ Job: ProcessVoterPdfImport
   ├─ Queue: database (or redis)
   ├─ Timeout: 3600 seconds
   └─ Retry: 3 attempts

3. PROCESSING
   ├─ Status: processing
   ├─ Parse: Extract text from 50 pages
   ├─ Detect: Pattern 1 found, Booth 123
   ├─ Extract: 2000 voters found
   └─ Batch: 20 batches × 100 voters

4. IMPORT (Batch 1/20)
   Transaction Start
   ├─ Voter 1: ABC1234567 → Insert
   ├─ Voter 2: XYZ9876543 → Insert
   ├─ Voter 3: DEF5555555 → Update (duplicate)
   ├─ ...
   └─ Voter 100: GHI7777777 → Insert
   Transaction Commit

5. COMPLETE
   ├─ Status: completed
   ├─ Total: 2000 voters
   ├─ Imported: 1985 voters
   ├─ Failed: 15 voters
   ├─ Duplicates: 12 voters
   └─ Time: 2.5 minutes

6. STATISTICS
   └─ JSON Summary:
      {
        "total_pages": 50,
        "total_voters": 2000,
        "imported_voters": 1985,
        "failed_voters": 15,
        "duplicate_voters": 12,
        "errors": [
          {"voter_id": "ABC1234568", "error": "Invalid year"}
        ]
      }


## Pattern Matching Flow

┌─────────────────────────────────────────────────────────────────────┐
│ PDF Text Line: "1 ABC1234567 John Doe M 1990 123 Main St"          │
└────────────────────────────┬────────────────────────────────────────┘
                             │
                             ▼
             ┌───────────────────────────────┐
             │   Pattern Detection           │
             └───────────┬───────────────────┘
                         │
         ┌───────────────┼───────────────┐
         │               │               │
         ▼               ▼               ▼
    ┌─────────┐    ┌─────────┐    ┌─────────┐
    │Pattern 1│    │Pattern 2│    │Pattern 3│
    │  Match! │    │  No     │    │  No     │
    └────┬────┘    └─────────┘    └─────────┘
         │
         ▼
┌─────────────────────────────────────┐
│  Extracted Data:                    │
│  ├─ serial_no: 1                    │
│  ├─ voter_id_number: ABC1234567     │
│  ├─ name: John Doe                  │
│  ├─ gender: M                       │
│  ├─ year_of_birth: 1990             │
│  ├─ address: 123 Main St            │
│  └─ booth_number: 123               │
└────────────┬────────────────────────┘
             │
             ▼
┌─────────────────────────────────────┐
│  Duplicate Check:                   │
│  WHERE voter_id_number = ABC1234567 │
└────────────┬────────────────────────┘
             │
     ┌───────┴────────┐
     │                │
     ▼                ▼
┌─────────┐      ┌─────────┐
│ Exists  │      │New Voter│
│ UPDATE  │      │ INSERT  │
└─────────┘      └─────────┘


## Status Lifecycle

┌──────────┐     ┌────────────┐     ┌───────────┐
│ PENDING  │────>│ PROCESSING │────>│ COMPLETED │
└──────────┘     └────────────┘     └───────────┘
                        │
                        │ (on error)
                        │
                        ▼
                 ┌──────────┐
                 │  FAILED  │
                 └────┬─────┘
                      │
                      │ (reprocess)
                      │
                      ▼
                 ┌──────────┐
                 │ PENDING  │
                 └──────────┘


## Error Handling Flow

┌─────────────────────────────────────────────────────────────────────┐
│                    Error Occurs During Processing                    │
└────────────────────────────┬────────────────────────────────────────┘
                             │
                ┌────────────┴────────────┐
                │                         │
         ▼                                ▼
┌──────────────────┐          ┌─────────────────────┐
│  Voter Error     │          │  Critical Error     │
│  (Single voter)  │          │  (Entire import)    │
└────────┬─────────┘          └──────────┬──────────┘
         │                               │
         ▼                               ▼
┌──────────────────┐          ┌─────────────────────┐
│ • Log error      │          │ • Rollback          │
│ • Continue next  │          │ • Mark as failed    │
│ • Track in JSON  │          │ • Store error msg   │
│ • Increment fail │          │ • Log full trace    │
└──────────────────┘          └─────────────────────┘


## Queue System Integration

┌─────────────────────────────────────────────────────────────────────┐
│                         Queue Worker                                 │
│                     php artisan queue:work                           │
└────────────────────────────┬────────────────────────────────────────┘
                             │
                             ▼
                ┌────────────────────────┐
                │   Job: ProcessVoter    │
                │   PdfImport            │
                └────────┬───────────────┘
                         │
                 ┌───────┴───────┐
                 │               │
          ▼                      ▼
┌───────────────┐      ┌────────────────┐
│   Success     │      │     Failed     │
│   • Complete  │      │   • Retry (3x) │
│   • Log stats │      │   • Log error  │
└───────────────┘      └────────┬───────┘
                                │
                         ┌──────┴──────┐
                         │             │
                  ▼                    ▼
          ┌──────────┐         ┌──────────┐
          │  Retry 1 │         │ Failed   │
          │  Retry 2 │         │ Table    │
          │  Retry 3 │         │          │
          └──────────┘         └──────────┘
```

## Component Interaction

```
Controller ──> Service ──> Parser ──> Database
    │              │          │           │
    │              │          │           └─> voters table
    │              │          │           └─> pdf_import_logs
    │              │          │
    │              │          └─> smalot/pdfparser (PDF → Text)
    │              │
    │              └─> Pattern Detection (Regex)
    │              └─> Batch Processing (100/batch)
    │              └─> Error Handling
    │
    └─> Job Queue ──> Background Processing
    └─> Response ──> JSON API
```

---

**Legend:**
- `→` : Flow direction
- `├─` : Component/feature
- `└─` : Final component/feature
- `▼` : Next step
- `✓` : Success/validation passed
